package easik.sketch.util;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;

import javax.swing.JOptionPane;


import org.jgrapht.graph.DefaultEdge;
import org.xml.sax.Attributes;
import org.xml.sax.helpers.DefaultHandler;

import easik.Easik;
import easik.sketch.attribute.EntityAttribute;
import easik.sketch.attribute.UniqueKey;
import easik.sketch.constraint.CommutativeDiagram;
import easik.sketch.constraint.Constraint;
import easik.sketch.constraint.ProductConstraint;
import easik.sketch.constraint.PullbackConstraint;
import easik.sketch.constraint.SumConstraint;
import easik.sketch.datatype.DataType;
import easik.sketch.document.DocumentInfo;
import easik.sketch.edge.SketchEdge;
import easik.sketch.path.SketchPath;
import easik.sketch.vertex.EntityNode;


/** 
 * The SketchHandler is the overloaded handler for reading the sketches in from
 * XML. There is very little error checking in here to deal with bad XML.
 * 
 * TODO: Implement schema checking
 * 
 * @author Rob Fletcher 2005
 * @author Kevin Green 2006
 * @author Vera Ranieri 2006
 * @version 2006-07-14 Vera Ranieri
 */
public class SketchHandler extends DefaultHandler {
	/**
	 * The entity nodes of the sketch, indexed by name
	 */
	private HashMap<String, EntityNode> _entityNodes;
	/**
	 * The edges of the sketch, indexed by name
	 */
	private HashMap<String, DefaultEdge> _edges;
	/**
	 * The paths of the sketch, indexed by name
	 */
	private HashMap<String, SketchPath> _allPaths;
	/**
	 * The constraints of the sketch
	 */
	private LinkedList<Constraint> _constraints;
	/**
	 * The data types defined for the sketch
	 */
	private ArrayList<DataType> _datatypes;
	/**
	 * The document information for the sketch
	 */
	private DocumentInfo _docInfo;
	/**
	 * Variable to store whether the respective platform is in use and defined.
	 */
	private boolean _useMySQL;
	/**
	 * Variable to store whether the respective platform is in use and defined.
	 */
	private boolean _useOracle;
	/**
	 * Variable to store whether the respective platform is in use and defined.
	 */
	private boolean _useDB2;
	/**
	 * Variable to store whether the respective platform is in use and defined.
	 */
	private boolean _useXML;
	/**
	 * Variable to store whether the respective platform is in use and defined.
	 */
	private boolean _useUserDefined;
	
	/**
	 * The edges involved in the current path being built from the XML file
	 */
	private LinkedList<SketchEdge> _curPath;
	/**
	 * The x-position of the current constraint being defined
	 */
	private int _curConstraintX = 0;
	/**
	 * The y-position of the current constraint being defined
	 */
	private int _curConstraintY = 0;
	/**
	 * The visibility of the current constraint
	 */
	private boolean _curConstraintVisible = true;
	/**
	 * Whether all constraints are visible
	 */
	private boolean _allConstraintsVisible;
	/**
	 * The paths involved in the current constraint being defined
	 */
	private ArrayList<SketchPath> _curConstraintPaths;
	/**
	 * The EntityAttributes involved in the current Unique key being defined
	 */
	private ArrayList<EntityAttribute> _curUniqueKeyAtts;
	/**
	 * Hash Map of the current node attributes, indexed by the name of the attribute
	 */
	private HashMap<String, EntityAttribute> _curNodeAtts;
	/**
	 * The name of the current unique key
	 */
	private String _curUniqueKeyName;
	/**
	 * The current entity node
	 */
	private EntityNode _newNode;
	/**
	 * The data type of the current attribute
	 */
	private DataType _curType;
	
	/**
	 * The name of the current entity or constraint
	 */
	private String _currNode;

	/**
	 * Default Constructor
	 */
	public SketchHandler() {
		_entityNodes = new HashMap<String, EntityNode>();
		_edges = new HashMap<String, DefaultEdge>();
		_constraints = new LinkedList<Constraint>();
		_datatypes = new ArrayList<DataType>();
		_allPaths = new HashMap<String, SketchPath>();
		_curPath = new LinkedList<SketchEdge>();
		_curConstraintPaths = new ArrayList<SketchPath>();
		_allConstraintsVisible = true;
		_docInfo = new DocumentInfo();
	}

	/**
	 * Returns HashMap of edges
	 * 
	 * @return HashMap of edges
	 */
	public HashMap<String, DefaultEdge> getEdges() {
		return _edges;
	}
	
	/**
	 * Returns HashMap of paths
	 * 
	 * @return HashMap of all paths
	 * @since 2006-05-29
	 */
	public HashMap<String, SketchPath> getPaths(){
		return _allPaths;
	}

	/**
	 * Returns HashMap of entities
	 * 
	 * @return HashMap of entities
	 */
	public HashMap<String, EntityNode> getEntities() {
		Iterator it = _entityNodes.keySet().iterator();
		while(it.hasNext()){
			EntityNode curNode = (EntityNode) _entityNodes.get(it.next());
			int numAttributes = curNode.getAttributes().size();
			for(int i=0; i<numAttributes; i++){
				((EntityAttribute)curNode.getAttributes().get(i)).setTypeFromTemp();
			}
		}
		return _entityNodes;
	}

	/**
	 * Returns LinkedList of constraints
	 * 
	 * @return LinkedList of constraints
	 */
	public LinkedList<Constraint> getConstraints() {
		return _constraints;
	}
	
	/**
	 * Returns ArrayList of data types
	 * 
	 * @return ArrayList of data types
	 */
	public ArrayList<DataType> getDataTypes(){
		return _datatypes;
	}
	
	/**
	 * Get the document information of the current sketch
	 * @return The DocumentInfo object associated with this sketch
	 */
	public DocumentInfo getDocumentInfo(){
		return _docInfo;
	}

	/**
	 * Overloaded method that is called any time the start of an element is found
	 * @param namespace @see org.xml.sax.helpers.DefaultHandler
	 * @param localName @see org.xml.sax.helpers.DefaultHandler
	 * @param qName @see org.xml.sax.helpers.DefaultHandler
	 * @param atts @see org.xml.sax.helpers.DefaultHandler
	 */
	public void startElement(String namespace, String localName, String qName, Attributes atts) {
		
		_currNode = qName;
		if (qName.equals("entity")) {
			String name = atts.getValue("name");
			int x = Integer.parseInt(atts.getValue("x"));
			int y = Integer.parseInt(atts.getValue("y"));
			_newNode = new EntityNode(name, x, y);
			_entityNodes.put(name, _newNode);
			_curNodeAtts = new HashMap<String, EntityAttribute>();
		} 
		else if (qName.equals("attribute")){
			EntityAttribute myAtt = new EntityAttribute(atts.getValue("name"), atts.getValue("attributeType"));
			_curNodeAtts.put(atts.getValue("name"), myAtt);
			_newNode.addAttribute(myAtt);
		}
		else if (qName.equals("uniqueKey")){
			_curUniqueKeyName = atts.getValue("name");
			_curUniqueKeyAtts = new ArrayList<EntityAttribute>();
		}
		else if (qName.equals("attref")){
			_curUniqueKeyAtts.add(_curNodeAtts.get(atts.getValue("name")));
		}
		else if (qName.equals("edge")) {
			SketchEdge newEdge;
			boolean injective = atts.getValue("injective").equals("true");
			EntityNode source =	(EntityNode) _entityNodes.get(atts.getValue("source"));
			EntityNode target =	(EntityNode) _entityNodes.get(atts.getValue("target"));
			String id = atts.getValue("id");
			newEdge = new SketchEdge(source, target, id, injective);
			_edges.put(id, newEdge);
		}
		else if (qName.equals("path")){
			_curPath = new LinkedList<SketchEdge>();
		}
		else if (qName.equals("edgeref")) {
			_curPath.add( (SketchEdge) _edges.get(atts.getValue("id")));
		}
		else if (qName.equals("sumconstraint") || qName.equals("pullbackconstraint")
					|| qName.equals("productconstraint") || qName.equals("commutativediagram")) 
		{
			_curConstraintX = Integer.parseInt(atts.getValue("x"));
			_curConstraintY = Integer.parseInt(atts.getValue("y"));
			_curConstraintVisible = ((String)atts.getValue("isVisible")).equals("true");
			_curConstraintPaths = new ArrayList<SketchPath>();
			_allConstraintsVisible = ((String)atts.getValue("isVisible")).equals("true");
		}
		else if (qName.equals("pathref")){
			_curConstraintPaths.add(_allPaths.get(atts.getValue("id")));
		}
		else if (qName.equals("datatypes")){
			//Check which platforms should be enabled
			if(atts.getValue("MySQL") != null && atts.getValue("MySQL").equals("true"))
				_useMySQL = true;
			else
				_useMySQL = false;
			
			if(atts.getValue("Oracle") != null && atts.getValue("Oracle").equals("true"))
				_useOracle = true;
			else
				_useOracle = false;
			
			if(atts.getValue("DB2") != null && atts.getValue("DB2").equals("true"))
				_useDB2 = true;
			else
				_useDB2 = false;
			
			if(atts.getValue("XML") != null && atts.getValue("XML").equals("true"))
				_useXML = true;
			else
				_useXML = false;
			
			if(atts.getValue("UserDefined") != null && atts.getValue("UserDefined").equals("true"))
				_useUserDefined = true;
			else
				_useUserDefined = false;
		}
		else if (qName.equals("datatype")){
			_curType = new DataType(atts.getValue("name"), atts.getValue("desc"));
			_datatypes.add(_curType);
		}
		else if (qName.equals("MySQL")){
			_curType.set_MySQL_type(atts.getValue("type"));
		}
		else if (qName.equals("Oracle")){
			_curType.set_Oracle_type(atts.getValue("type"));
		}
		else if (qName.equals("DB2")){
			_curType.set_DB2_type(atts.getValue("type"));
		}
		else if (qName.equals("XML")){
			_curType.set_XML_type(atts.getValue("type"));
		}
		else if (qName.equals("UserDefined")){
			_curType.set_user_type(atts.getValue("type"));
		}
	}

	/**
	 * Overloaded method that is called any time the end of an element is found
	 * @param uri @see org.xml.sax.helpers.DefaultHandler
	 * @param localName @see org.xml.sax.helpers.DefaultHandler
	 * @param qName @see org.xml.sax.helpers.DefaultHandler
	 */
	public void endElement(String uri, String localName, String qName) {
		_currNode = null;
		if (qName.equals("uniqueKey")){
			UniqueKey newKey = new UniqueKey(_newNode, _curUniqueKeyAtts, _curUniqueKeyName);
			_newNode.addUniqueKey(newKey);
		}
		else if (qName.equals("sumconstraint")) {
			if(SumConstraint.isSumConstraint(_curConstraintPaths)){
				_constraints.add(new SumConstraint(_curConstraintPaths, _curConstraintX, _curConstraintY, _curConstraintVisible));
			}
			else{
				JOptionPane.showMessageDialog( Easik.getInstance().getFrame(),
						"An error occured while loading a sum constraint.\n" 
						+"Error on sum constraint between tables\n"
						+ Constraint.getTablesInvolvedForError(_curConstraintPaths),
						"Error",
						JOptionPane.ERROR_MESSAGE);
			}
		} 
		else if (qName.equals("pullbackconstraint")) {
			if(PullbackConstraint.isPullbackConstraint(_curConstraintPaths)){
				_constraints.add(new PullbackConstraint(_curConstraintPaths, _curConstraintX, _curConstraintY, _curConstraintVisible));
			}
			else{
				JOptionPane.showMessageDialog( Easik.getInstance().getFrame(),
						"An error occured while loading a constraint.\n "
						+"Error on pullback between tables\n"
						+ Constraint.getTablesInvolvedForError(_curConstraintPaths),
						"Error",
						JOptionPane.ERROR_MESSAGE);
			}
		} 
		else if (qName.equals("productconstraint")) {
			if(ProductConstraint.isProductConstraint(_curConstraintPaths)){
				_constraints.add(new ProductConstraint(_curConstraintPaths, _curConstraintX, _curConstraintY, _curConstraintVisible));
			}
			else{
				JOptionPane.showMessageDialog( Easik.getInstance().getFrame(),
						"An error occured while loading a constraint.\n"
						+"Error on product constraint between tables\n"
						+Constraint.getTablesInvolvedForError(_curConstraintPaths),
						"Error",
						JOptionPane.ERROR_MESSAGE);
			}
		}
		else if (qName.equals("commutativediagram")){
			if(_curConstraintPaths.size() == 2 && 
					CommutativeDiagram.isCommutativeDiagram(_curConstraintPaths.get(0), _curConstraintPaths.get(1))){
				_constraints.add(new CommutativeDiagram(_curConstraintPaths, _curConstraintX, _curConstraintY, _curConstraintVisible));
			}
			else{
				JOptionPane.showMessageDialog( Easik.getInstance().getFrame(),
						"An error occured while loading a constraint.\n"
						+"Error on commutative diagram between tables\n"
						+ Constraint.getTablesInvolvedForError(_curConstraintPaths),
						"Error",
						JOptionPane.ERROR_MESSAGE);
			}
		}
		else if (qName.equals("path")){
			SketchPath myPath = new SketchPath(_curPath);
			_allPaths.put(myPath.getId(), myPath);
		}
		else if(qName.equals("constraints")){
			Easik.getInstance().getFrame().setShowConstraints(_allConstraintsVisible);
		}
	}
	
	/**
	 * @see org.xml.sax.helpers.DefaultHandler
	 * @param ch @see org.xml.sax.helpers.DefaultHandler
	 * @param start @see org.xml.sax.helpers.DefaultHandler
	 * @param length @see org.xml.sax.helpers.DefaultHandler
	 */
	public void characters(char[] ch, int start, int length){
		if(_currNode == null)
			return;
		if(_currNode.equals("title")){
			_docInfo.setName(new String (ch, start, length));
		}
		else if (_currNode.equals("author"))
			_docInfo.addAuthor(new String (ch, start, length));
		else if (_currNode.equals("description"))
			_docInfo.setDesc(new String(ch, start, length));
		else if (_currNode.equals("creationDate"))
			_docInfo.setCreationDate(new String(ch, start, length));
		else if (_currNode.equals("lastModificationDate"))
			_docInfo.setModificationDate(new String(ch, start, length));
	}

	/**
	 * @return Returns the _useDB2.
	 */
	public boolean is_useDB2() {
		return _useDB2;
	}

	/**
	 * @return Returns the _useMySQL.
	 */
	public boolean is_useMySQL() {
		return _useMySQL;
	}

	/**
	 * @return Returns the _useOracle.
	 */
	public boolean is_useOracle() {
		return _useOracle;
	}

	/**
	 * @return Returns the _useUserDefined.
	 */
	public boolean is_useUserDefined() {
		return _useUserDefined;
	}

	/**
	 * @return Returns the _useXML.
	 */
	public boolean is_useXML() {
		return _useXML;
	}
}
